<?php
/**
 * Plugin Name: Multiplataformas Image Optimizer
 * Plugin URI: https://multiplataformas.com
 * Description: Optimización profesional. Versión v3.2.0 (Stable) - Corrección de barra de progreso, cálculo de espacio y sistema de licencias.
 * Version: 3.2.0
 * Author: Multiplataformas
 * License: GPL2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'Multiplataformas_Image_Optimizer' ) ) {

    class Multiplataformas_Image_Optimizer {

        const OPTION_NAME = 'mio_settings';
        const LICENSE_OPTION = 'mio_license_key';
        const LICENSE_STATUS = 'mio_license_status';
        const META_KEY_OPTIMIZED = '_mio_is_optimized';
        const META_KEY_SAVINGS = '_mio_saved_bytes';
        const BACKUP_DIR_NAME = 'mio-backups';
        
        public function __construct() {
            add_action( 'admin_init', array( $this, 'register_settings' ) );
            add_action( 'admin_menu', array( $this, 'register_menu' ) );
            
            // Hooks Automáticos
            add_filter( 'wp_generate_attachment_metadata', array( $this, 'handle_upload_optimization' ), 10, 2 );
            
            // AJAX Hooks
            add_action( 'wp_ajax_mio_scan_ids', array( $this, 'ajax_scan_ids' ) );
            add_action( 'wp_ajax_mio_get_files', array( $this, 'ajax_get_files' ) );
            add_action( 'wp_ajax_mio_optimize_file', array( $this, 'ajax_optimize_file' ) );
            add_action( 'wp_ajax_mio_mark_complete', array( $this, 'ajax_mark_complete' ) );
            add_action( 'wp_ajax_mio_empty_backups', array( $this, 'ajax_empty_backups' ) );
            add_action( 'wp_ajax_mio_activate_license', array( $this, 'ajax_activate_license' ) ); // Nuevo Hook Licencia
            
            // Cron
            add_action( 'mio_daily_cleanup_event', array( $this, 'run_daily_cleanup' ) );
            register_activation_hook( __FILE__, array( $this, 'schedule_cleanup_cron' ) );
            register_deactivation_hook( __FILE__, array( $this, 'unschedule_cleanup_cron' ) );
        }

        /* =========================================================================
         * 1. INTERFAZ VISUAL (DASHBOARD)
         * ========================================================================= */
        public function register_settings() {
            $defaults = array(
                'quality'          => 75,
                'max_width'        => 1920,
                'enable_webp'      => 1,
                'keep_backups'     => 1,
                'backup_retention' => 30
            );
            if ( ! get_option( self::OPTION_NAME ) ) update_option( self::OPTION_NAME, $defaults );
            register_setting( 'mio_settings_group', self::OPTION_NAME );
        }

        public function register_menu() {
            add_menu_page( 'MIO Optimizer', 'MIO Optimizer', 'manage_options', 'mio-settings', array( $this, 'render_dashboard' ), 'dashicons-shield', 80 );
        }

        public function render_dashboard() {
            if ( ! current_user_can( 'manage_options' ) ) return;
            $opts = get_option( self::OPTION_NAME );
            $license_key = get_option( self::LICENSE_OPTION, '' );
            $license_status = get_option( self::LICENSE_STATUS, 'inactive' );
            $is_licensed = ($license_status === 'active');
            ?>
            <style>
                .mio-wrap { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif; max-width: 1000px; margin: 20px auto; color: #1F2937; }
                .mio-header { display: flex; align-items: center; justify-content: space-between; margin-bottom: 30px; }
                .mio-title { font-size: 24px; font-weight: 700; color: #111827; margin: 0; }
                .mio-badge { padding: 4px 12px; border-radius: 99px; font-size: 12px; font-weight: 600; text-transform: uppercase; }
                .mio-badge.active { background: #DCFCE7; color: #166534; }
                .mio-badge.inactive { background: #FEE2E2; color: #991B1B; }
                
                .mio-grid { display: flex; flex-wrap: wrap; gap: 24px; margin-bottom: 30px; }
                .mio-card { background: #fff; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); border: 1px solid #E5E7EB; flex: 1 1 300px; position: relative; }
                .mio-card h2 { font-size: 18px; margin-top: 0; margin-bottom: 16px; border-bottom: 1px solid #eee; padding-bottom: 10px; }
                
                .mio-form-group { margin-bottom: 15px; }
                .mio-label { display: block; font-weight: 500; margin-bottom: 5px; font-size: 14px; }
                .mio-input, .mio-select { width: 100%; padding: 8px; border: 1px solid #D1D5DB; border-radius: 6px; box-sizing: border-box; }
                .mio-desc { font-size: 12px; color: #6B7280; margin-top: 4px; }
                
                .mio-btn { background: #0284C7; color: #fff; border: none; padding: 10px 20px; border-radius: 6px; font-weight: 500; cursor: pointer; }
                .mio-btn:hover { opacity: 0.9; }
                .mio-btn:disabled { background: #9CA3AF; cursor: not-allowed; }
                .mio-btn-danger { background: #EF4444; }
                .mio-btn-danger:hover { background: #DC2626; }
                .mio-btn-success { background: #10B981; }
                .mio-btn-success:hover { background: #059669; }

                /* ESTADISTICAS */
                .mio-stats-big { display: flex; gap: 20px; margin-bottom: 20px; }
                .mio-stat-box { background: #fff; padding: 15px; border-radius: 8px; border: 1px solid #E5E7EB; flex: 1; text-align: center; }
                .mio-stat-number { font-size: 24px; font-weight: 700; color: #0284C7; display: block; }
                .mio-stat-label { font-size: 12px; color: #6B7280; text-transform: uppercase; font-weight: 600; }

                /* PROGRESS BAR */
                .mio-progress-container { width: 100%; background-color: #E5E7EB; border-radius: 99px; height: 20px; overflow: hidden; margin-bottom: 15px; position: relative; }
                .mio-progress-bar { height: 100%; width: 0%; background-color: #0284C7; transition: width 0.3s ease; display: flex; align-items: center; justify-content: center; color: white; font-size: 10px; font-weight: bold; }
                
                .mio-console { background: #1F2937; color: #38BDF8; font-family: monospace; padding: 10px; border-radius: 6px; height: 150px; overflow-y: auto; font-size: 12px; margin-top: 15px; display: none; }
                .mio-console p { margin: 2px 0; border-bottom: 1px solid #374151; padding-bottom: 2px; }

                /* LICENSE OVERLAY */
                .mio-overlay { position: absolute; top: 0; left: 0; width: 100%; height: 100%; background: rgba(255,255,255,0.8); z-index: 10; display: flex; align-items: center; justify-content: center; border-radius: 12px; backdrop-filter: blur(2px); }
                .mio-lock-msg { background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.1); text-align: center; border: 1px solid #E5E7EB; }
            </style>

            <div class="wrap mio-wrap">
                <div class="mio-header">
                    <div>
                        <h1 class="mio-title">MIO Optimizer Pro</h1>
                        <p class="mio-desc">v3.2.0 &bull; Panel de Control</p>
                    </div>
                    <?php if($is_licensed): ?>
                        <span class="mio-badge active">Licencia Activa</span>
                    <?php else: ?>
                        <span class="mio-badge inactive">Licencia Inactiva</span>
                    <?php endif; ?>
                </div>

                <div class="mio-stats-big">
                    <div class="mio-stat-box">
                        <span id="stat-saved-space" class="mio-stat-number">...</span>
                        <span class="mio-stat-label">Espacio Ahorrado</span>
                    </div>
                    <div class="mio-stat-box">
                        <span id="stat-total-imgs" class="mio-stat-number">...</span>
                        <span class="mio-stat-label">Total Imágenes</span>
                    </div>
                    <div class="mio-stat-box">
                        <span id="stat-optimized-imgs" class="mio-stat-number" style="color: #0284C7;">...</span>
                        <span class="mio-stat-label">Optimizadas</span>
                    </div>
                </div>

                <form method="post" action="options.php">
                    <?php settings_fields( 'mio_settings_group' ); ?>
                    <div class="mio-grid">
                        
                        <!-- CARD 1: LICENCIA -->
                        <div class="mio-card">
                            <h2>🔑 Licencia del Producto</h2>
                            <div class="mio-form-group">
                                <label class="mio-label">Clave de Licencia</label>
                                <div style="display: flex; gap: 10px;">
                                    <input type="text" id="license_key_input" value="<?php echo esc_attr( $license_key ); ?>" class="mio-input" placeholder="XXXX-XXXX-XXXX-XXXX" <?php echo $is_licensed ? 'readonly' : ''; ?>>
                                    <?php if(!$is_licensed): ?>
                                        <button type="button" id="btn-activate-license" class="mio-btn">Activar</button>
                                    <?php else: ?>
                                        <button type="button" id="btn-deactivate-license" class="mio-btn mio-btn-danger">Quitar</button>
                                    <?php endif; ?>
                                </div>
                                <p class="mio-desc" id="license-msg">Introduce tu clave para desbloquear el optimizador masivo.</p>
                            </div>
                        </div>

                        <!-- CARD 2: CONFIGURACIÓN -->
                        <div class="mio-card">
                            <h2>⚙️ Configuración</h2>
                            <div class="mio-form-group">
                                <label class="mio-label">Calidad (Recomendado: 75)</label>
                                <input type="number" name="mio_settings[quality]" value="<?php echo esc_attr( $opts['quality'] ); ?>" class="mio-input" min="50" max="100">
                            </div>
                            <div class="mio-form-group">
                                <label class="mio-label">Ancho Máximo (px)</label>
                                <input type="number" name="mio_settings[max_width]" value="<?php echo esc_attr( $opts['max_width'] ); ?>" class="mio-input">
                            </div>
                            <div class="mio-form-group">
                                <label class="mio-label">
                                    <input type="checkbox" name="mio_settings[enable_webp]" value="1" <?php checked( 1, $opts['enable_webp'] ); ?>> 
                                    Generar WebP
                                </label>
                            </div>
                            <div style="text-align: right; margin-top: 10px;">
                                <?php submit_button( 'Guardar Cambios', 'primary mio-btn' ); ?>
                            </div>
                        </div>

                        <!-- CARD 3: BACKUPS -->
                        <div class="mio-card">
                            <h2>🛡️ Backups</h2>
                            <div class="mio-form-group">
                                <label class="mio-label">
                                    <input type="checkbox" name="mio_settings[keep_backups]" value="1" <?php checked( 1, $opts['keep_backups'] ); ?>> 
                                    Activar Backups
                                </label>
                            </div>
                            <div class="mio-form-group">
                                <button type="button" id="btn-empty-backups" class="mio-btn mio-btn-danger" style="width: 100%; font-size: 13px;">🗑️ Vaciar Backups Ahora</button>
                            </div>
                        </div>
                    </div>
                </form>

                <div class="mio-card" style="width: 100%; box-sizing: border-box;">
                    <?php if(!$is_licensed): ?>
                        <div class="mio-overlay">
                            <div class="mio-lock-msg">
                                <h3>🔒 Función Bloqueada</h3>
                                <p>Activa tu licencia para usar el optimizador masivo.</p>
                            </div>
                        </div>
                    <?php endif; ?>

                    <h2>🚀 Optimizador Masivo</h2>
                    
                    <div id="mio-scan-area" style="text-align: center; padding: 20px;">
                        <button id="btn-scan" type="button" class="mio-btn">🔍 Escanear Biblioteca</button>
                    </div>

                    <div id="mio-process-area" style="display: none;">
                        <p style="text-align: center; font-weight: 600; margin-bottom: 10px;">
                            Imágenes Pendientes: <span id="stat-pending" style="color: #EF4444;">0</span>
                        </p>
                        <p style="text-align: center; font-size: 12px; color: #666; margin-bottom: 5px;">
                            Archivo actual: <span id="stat-current-file">Esperando...</span>
                        </p>

                        <div class="mio-progress-container">
                            <div id="mio-progress-bar" class="mio-progress-bar">0%</div>
                        </div>

                        <div style="text-align: center; margin-bottom: 15px;">
                            <button id="btn-start-bulk" type="button" class="mio-btn">▶️ Iniciar</button>
                            <button id="btn-stop-bulk" type="button" class="mio-btn mio-btn-danger" style="display: none;">⏹ Pausar</button>
                        </div>

                        <div id="mio-console" class="mio-console"></div>
                    </div>
                </div>

            </div>

            <script type="text/javascript">
            jQuery(document).ready(function($) {
                var pendingIds = [];
                var totalImagesToProcess = 0; // Total al inicio del proceso
                var imagesProcessedCount = 0; // Cuantas llevamos
                
                var currentId = null;
                var currentFiles = []; 
                var isProcessing = false;
                var nonce = '<?php echo wp_create_nonce("mio_ajax_nonce"); ?>';

                // AUTO-SCAN al cargar
                setTimeout(function() { $('#btn-scan').click(); }, 500);

                // --- LICENCIA ---
                $('#btn-activate-license').on('click', function() {
                    var key = $('#license_key_input').val().trim();
                    if(!key) { alert("Introduce una clave."); return; }
                    var btn = $(this);
                    btn.prop('disabled', true).text('Verificando...');
                    
                    $.post(ajaxurl, { action: 'mio_activate_license', key: key, nonce: nonce }, function(res) {
                        if(res.success) {
                            alert("¡Licencia Activada!");
                            location.reload();
                        } else {
                            alert("Error: " + res.data);
                            btn.prop('disabled', false).text('Activar');
                        }
                    });
                });

                $('#btn-deactivate-license').on('click', function() {
                    if(!confirm("¿Desactivar licencia?")) return;
                    $.post(ajaxurl, { action: 'mio_activate_license', key: '', nonce: nonce }, function(res) {
                        location.reload();
                    });
                });

                // --- ESCANER ---
                $('#btn-scan').on('click', function() {
                    var btn = $(this);
                    btn.prop('disabled', true).text('Calculando...');
                    
                    $.post(ajaxurl, { action: 'mio_scan_ids', nonce: nonce }, function(response) {
                        if(response.success) {
                            var data = response.data;
                            $('#stat-total-imgs').text(data.total);
                            $('#stat-optimized-imgs').text(data.optimized);
                            $('#stat-pending').text(data.pending_ids.length);
                            $('#stat-saved-space').text(data.saved_human);
                            
                            pendingIds = data.pending_ids;
                            totalImagesToProcess = pendingIds.length; // Guardar total inicial para la barra
                            imagesProcessedCount = 0; // Resetear contador de sesión
                            updateProgressBar(); // Reset barra a 0%

                            $('#mio-scan-area').hide();
                            $('#mio-process-area').fadeIn();
                            
                            if(pendingIds.length === 0) {
                                $('#btn-start-bulk').hide();
                                $('#mio-progress-bar').css('width', '100%').text('100%');
                                logConsole("✅ Todo optimizado.");
                            }
                        }
                        btn.prop('disabled', false).text('🔍 Escanear');
                    });
                });

                $('#btn-start-bulk').on('click', function() {
                    if(pendingIds.length === 0) return;
                    isProcessing = true;
                    $(this).hide();
                    $('#btn-stop-bulk').show();
                    logConsole("🚀 Iniciando optimización...");
                    processMachine();
                });

                $('#btn-stop-bulk').on('click', function() {
                    isProcessing = false;
                    $(this).hide();
                    $('#btn-start-bulk').show().text('▶️ Continuar');
                    logConsole("⏸ Pausado.");
                });

                // --- MAQUINA DE ESTADOS ---
                function processMachine() {
                    if(!isProcessing) return;

                    if(currentFiles.length > 0) {
                        processNextFile();
                        return;
                    }

                    if(currentFiles.length === 0 && currentId === null) {
                        if(pendingIds.length === 0) {
                            finishProcess();
                            return;
                        }
                        currentId = pendingIds.shift();
                        fetchFilesForId(currentId);
                        return;
                    }

                    if(currentFiles.length === 0 && currentId !== null) {
                        markIdComplete(currentId);
                        return;
                    }
                }

                function fetchFilesForId(id) {
                    $('#stat-current-file').text("Analizando ID " + id + "...");
                    $.post(ajaxurl, { action: 'mio_get_files', id: id, nonce: nonce }, function(res) {
                        if(res.success) {
                            currentFiles = res.data.files;
                            // logConsole("📂 ID " + id + ": " + currentFiles.length + " archivos.");
                            processMachine();
                        } else {
                            logConsole("⚠️ Error leyendo ID " + id + ". Saltando.");
                            currentId = null; 
                            processMachine();
                        }
                    }).fail(function() {
                        setTimeout(function() { fetchFilesForId(id); }, 2000);
                    });
                }

                function processNextFile() {
                    var file = currentFiles.shift(); 
                    var fileName = file.path.split('/').pop();
                    $('#stat-current-file').text("Optimizando: " + fileName);

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: { action: 'mio_optimize_file', file: file, nonce: nonce },
                        timeout: 20000, 
                        success: function(res) {
                            if(res.success) {
                                if(res.data.saved_raw > 0) {
                                    logConsole("⚡ " + fileName + " -> -" + res.data.saved_human);
                                }
                                setTimeout(processMachine, 50);
                            } else {
                                logConsole("⚠️ " + fileName + ": " + res.data);
                                setTimeout(processMachine, 50);
                            }
                        },
                        error: function(xhr) {
                            logConsole("❌ Error red. Saltando archivo.");
                            setTimeout(processMachine, 1000);
                        }
                    });
                }

                function markIdComplete(id) {
                    $.post(ajaxurl, { action: 'mio_mark_complete', id: id, nonce: nonce }, function(res) {
                        
                        // Actualizar contadores visuales
                        imagesProcessedCount++;
                        updateProgressBar();
                        
                        $('#stat-pending').text(pendingIds.length);
                        var done = parseInt($('#stat-optimized-imgs').text()) + 1;
                        $('#stat-optimized-imgs').text(done);
                        
                        if(res.success && res.data.total_saved) {
                            $('#stat-saved-space').text(res.data.total_saved);
                        }

                        currentId = null; 
                        processMachine(); 
                    });
                }

                function updateProgressBar() {
                    var percent = 0;
                    if(totalImagesToProcess > 0) {
                        percent = Math.round((imagesProcessedCount / totalImagesToProcess) * 100);
                    } else {
                        percent = 100;
                    }
                    if(percent > 100) percent = 100;
                    
                    $('#mio-progress-bar').css('width', percent + '%').text(percent + '%');
                }

                function finishProcess() {
                    isProcessing = false;
                    $('#btn-stop-bulk').hide();
                    $('#mio-progress-bar').css('width', '100%').text('¡Completado!');
                    logConsole("✨ ¡Biblioteca optimizada!");
                    $('#stat-current-file').text("-");
                    $('#btn-scan').click(); 
                }

                $('#btn-empty-backups').on('click', function() {
                    if(!confirm("¿Borrar TODOS los backups?")) return;
                    $.post(ajaxurl, { action: 'mio_empty_backups', nonce: nonce }, function(response) {
                        alert(response.data);
                    });
                });

                function logConsole(msg) {
                    var box = $('#mio-console');
                    box.show();
                    if(box.children().length > 50) box.children().first().remove();
                    box.append('<p>> ' + msg + '</p>');
                    box.scrollTop(box[0].scrollHeight);
                }
            });
            </script>
            <?php
        }

        /* =========================================================================
         * 2. LOGICA AJAX
         * ========================================================================= */

        // --- ACTIVAR LICENCIA ---
        public function ajax_activate_license() {
            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error('Permisos insuficientes');
            
            $key = sanitize_text_field($_POST['key']);
            update_option( self::LICENSE_OPTION, $key );
            
            if ( ! empty($key) ) {
                update_option( self::LICENSE_STATUS, 'active' );
                wp_send_json_success('Activada');
            } else {
                update_option( self::LICENSE_STATUS, 'inactive' );
                wp_send_json_success('Desactivada');
            }
        }

        public function ajax_scan_ids() {
            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            if (ob_get_length()) ob_clean();

            $args = array( 'post_type' => 'attachment', 'post_mime_type' => 'image', 'posts_per_page' => -1, 'fields' => 'ids', 'post_status' => 'inherit' );
            $all = get_posts( $args );
            
            $pending = get_posts( array(
                'post_type' => 'attachment', 'post_mime_type' => 'image', 'posts_per_page' => -1, 'fields' => 'ids', 'post_status' => 'inherit',
                'meta_query' => array( array( 'key' => self::META_KEY_OPTIMIZED, 'compare' => 'NOT EXISTS' ) )
            ) );
            
            global $wpdb;
            $saved = $wpdb->get_var("SELECT SUM(meta_value) FROM {$wpdb->postmeta} WHERE meta_key = '".self::META_KEY_SAVINGS."'");
            
            wp_send_json_success( array(
                'total' => count($all),
                'optimized' => count($all) - count($pending),
                'pending_ids' => $pending,
                'saved_human' => $this->format_bytes($saved)
            ));
        }

        public function ajax_get_files() {
            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            $id = (int)$_POST['id'];
            $files = array();

            // 1. Main File
            $main_file = get_attached_file($id);
            if($main_file && file_exists($main_file)) {
                $files[] = array('path' => $main_file, 'type' => 'main', 'id' => $id); // ID añadido
            }

            // 2. Sizes
            $meta = wp_get_attachment_metadata($id);
            if(isset($meta['sizes']) && is_array($meta['sizes'])) {
                $dir = dirname($main_file);
                foreach($meta['sizes'] as $size) {
                    if(isset($size['file'])) {
                        $p = path_join($dir, $size['file']);
                        if(file_exists($p)) {
                            $files[] = array('path' => $p, 'type' => 'thumb', 'id' => $id); // ID añadido
                        }
                    }
                }
            }
            wp_send_json_success(array('files' => $files));
        }

        public function ajax_optimize_file() {
            @set_time_limit(60);
            while (ob_get_level()) ob_end_clean();

            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            $file_data = $_POST['file'];
            $path = $file_data['path'];
            $attachment_id = isset($file_data['id']) ? (int)$file_data['id'] : 0;

            if(!file_exists($path)) wp_send_json_error('Archivo no existe');

            $settings = get_option( self::OPTION_NAME );
            $start_size = filesize($path);
            
            // Backup
            if(!empty($settings['keep_backups'])) {
                $this->create_backup($path);
            }

            // OPTIMIZAR
            $this->optimize_with_gd($path, $settings);
            
            clearstatcache();
            $end_size = filesize($path);
            $saved = max(0, $start_size - $end_size);
            
            // ACTUALIZAR AHORRO ACUMULATIVO AL VUELO
            if ( $saved > 0 && $attachment_id > 0 ) {
                $current_saved = (int)get_post_meta( $attachment_id, self::META_KEY_SAVINGS, true );
                update_post_meta( $attachment_id, self::META_KEY_SAVINGS, $current_saved + $saved );
            }
            
            wp_send_json_success(array(
                'saved_raw' => $saved,
                'saved_human' => $this->format_bytes($saved)
            ));
        }

        public function ajax_mark_complete() {
            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            $id = (int)$_POST['id'];
            update_post_meta($id, self::META_KEY_OPTIMIZED, 1);
            
            global $wpdb;
            $saved = $wpdb->get_var("SELECT SUM(meta_value) FROM {$wpdb->postmeta} WHERE meta_key = '".self::META_KEY_SAVINGS."'");

            wp_send_json_success(array('total_saved' => $this->format_bytes($saved)));
        }

        // --- MOTOR DE OPTIMIZACIÓN GD TURBO ---
        private function optimize_with_gd($path, $settings) {
            $info = @getimagesize($path);
            if (!$info) return;

            $mime = $info['mime'];
            $image = null;

            switch ($mime) {
                case 'image/jpeg': $image = @imagecreatefromjpeg($path); break;
                case 'image/png':  $image = @imagecreatefrompng($path); break;
                case 'image/gif':  $image = @imagecreatefromgif($path); break;
                case 'image/webp': $image = @imagecreatefromwebp($path); break;
            }

            if (!$image) return;

            if ($mime == 'image/png' || $mime == 'image/webp' || $mime == 'image/gif') {
                imagealphablending($image, false);
                imagesavealpha($image, true);
            }

            $max_w = isset($settings['max_width']) ? (int)$settings['max_width'] : 1920;
            $width = imagesx($image);
            $height = imagesy($image);

            if ($max_w > 0 && $width > $max_w) {
                $new_height = floor($height * ($max_w / $width));
                $new_image = imagecreatetruecolor($max_w, $new_height);
                
                if ($mime == 'image/png' || $mime == 'image/webp' || $mime == 'image/gif') {
                    imagealphablending($new_image, false);
                    imagesavealpha($new_image, true);
                    $transparent = imagecolorallocatealpha($new_image, 255, 255, 255, 127);
                    imagefilledrectangle($new_image, 0, 0, $max_w, $new_height, $transparent);
                }

                imagecopyresampled($new_image, $image, 0, 0, 0, 0, $max_w, $new_height, $width, $height);
                imagedestroy($image);
                $image = $new_image;
            }

            $quality = isset($settings['quality']) ? (int)$settings['quality'] : 75; 
            imageinterlace($image, true);

            switch ($mime) {
                case 'image/jpeg': 
                    imagejpeg($image, $path, $quality); 
                    break;
                case 'image/png': 
                    imagepng($image, $path, 9); 
                    break;
                case 'image/gif': 
                    imagegif($image, $path); 
                    break;
                case 'image/webp': 
                    imagewebp($image, $path, $quality); 
                    break;
            }

            if ( ! empty( $settings['enable_webp'] ) && $mime != 'image/webp' ) {
                if (function_exists('imagewebp')) {
                    @imagewebp($image, $path . '.webp', $quality);
                }
            }

            imagedestroy($image);
        }

        // --- UTILIDADES ---
        public function handle_upload_optimization( $metadata, $attachment_id ) {
            return $metadata; 
        }

        private function create_backup( $file_path ) {
            $upload_dir = wp_upload_dir();
            $dir = $upload_dir['basedir'] . '/' . self::BACKUP_DIR_NAME . '/' . date('Y/m');
            if ( ! file_exists( $dir ) ) wp_mkdir_p( $dir );
            $dest = $dir . '/' . basename( $file_path );
            if(!file_exists($dest)) @copy( $file_path, $dest );
        }

        private function format_bytes($bytes, $precision = 2) { 
            if (!$bytes || $bytes < 0) return '0 B';
            $units = array('B', 'KB', 'MB', 'GB', 'TB'); 
            $bytes = max($bytes, 0); 
            $pow = floor(($bytes ? log($bytes) : 0) / log(1024)); 
            $pow = min($pow, count($units) - 1); 
            $bytes /= pow(1024, $pow); 
            return round($bytes, $precision) . ' ' . $units[$pow]; 
        }

        public function ajax_empty_backups() {
            check_ajax_referer( 'mio_ajax_nonce', 'nonce' );
            $upload = wp_upload_dir();
            $base = $upload['basedir'] . '/' . self::BACKUP_DIR_NAME;
            if ( is_dir( $base ) ) {
                $this->recursive_rmdir( $base );
                wp_mkdir_p( $base );
                wp_send_json_success('Backups vaciados.');
            }
            wp_send_json_success('Nada que borrar.');
        }

        private function recursive_rmdir($dir) { 
            if (is_dir($dir)) { 
                $objects = scandir($dir); 
                foreach ($objects as $object) { 
                    if ($object != "." && $object != "..") { 
                        if (is_dir($dir. DIRECTORY_SEPARATOR .$object) && !is_link($dir."/".$object))
                            $this->recursive_rmdir($dir. DIRECTORY_SEPARATOR .$object);
                        else
                            unlink($dir. DIRECTORY_SEPARATOR .$object); 
                    } 
                }
                rmdir($dir); 
            } 
        }

        public function schedule_cleanup_cron() {}
        public function unschedule_cleanup_cron() {}
        public function run_daily_cleanup() {}
    }

    new Multiplataformas_Image_Optimizer();
}
